<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Payment_model extends CI_Model {
    
    private $table_payment = 'payment_log';
    private $table_subscription = 'admin_subscription';
    private $table_admin = 'admin';
    
    public function __construct() {
        parent::__construct();
        $this->load->database();
    }
    
    /**
     * Create payment record
     */
    public function create_payment($data) {
        $this->db->insert($this->table_payment, $data);
        return $this->db->insert_id();
    }
    
    /**
     * Update payment status
     */
    public function update_payment_status($order_id, $data) {
        $this->db->where('order_id', $order_id);
        return $this->db->update($this->table_payment, $data);
    }
    
    /**
     * Get payment by order ID
     */
    public function get_payment_by_order_id($order_id) {
        return $this->db->get_where($this->table_payment, ['order_id' => $order_id])->row_array();
    }
    
    /**
     * Get payments by admin ID
     */
    public function get_payments_by_admin($admin_id, $limit = 10, $offset = 0) {
        $this->db->where('id_admin', $admin_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit, $offset);
        return $this->db->get($this->table_payment)->result_array();
    }
    
    /**
     * Get pending payments count
     */
    public function get_pending_payments_count($admin_id) {
        $this->db->where('id_admin', $admin_id);
        $this->db->where('transaction_status', 'pending');
        return $this->db->count_all_results($this->table_payment);
    }
    
    /**
     * Get admin subscription
     */
    public function get_admin_subscription($admin_id) {
        $this->db->where('id_admin', $admin_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit(1);
        return $this->db->get($this->table_subscription)->row_array();
    }
    
    /**
     * Create subscription
     */
    public function create_subscription($data) {
        $this->db->insert($this->table_subscription, $data);
        return $this->db->insert_id();
    }
    
    /**
     * Update subscription
     */
    public function update_subscription($id, $data) {
        $this->db->where('id_subscription', $id);
        return $this->db->update($this->table_subscription, $data);
    }
    
    /**
     * Check if admin has active subscription
     */
    public function check_admin_subscription($admin_id) {
        $this->db->where('id_admin', $admin_id);
        $this->db->where('status', 'active');
        $this->db->where('end_date >', date('Y-m-d H:i:s'));
        $count = $this->db->count_all_results($this->table_subscription);
        return $count > 0;
    }
    
    /**
     * Get active subscription end date
     */
    public function get_active_subscription_end_date($admin_id) {
        $this->db->select('end_date');
        $this->db->where('id_admin', $admin_id);
        $this->db->where('status', 'active');
        $this->db->where('end_date >', date('Y-m-d H:i:s'));
        $this->db->order_by('end_date', 'DESC');
        $this->db->limit(1);
        $result = $this->db->get($this->table_subscription)->row_array();
        
        return $result ? $result['end_date'] : null;
    }
    
    /**
     * Get admin data
     */
    public function get_admin_data($admin_id) {
        return $this->db->get_where($this->table_admin, ['id' => $admin_id])->row_array();
    }
    
    /**
     * Activate chat feature
     */
    public function activate_chat_feature($admin_id, $duration_days = 30) {
        $current_date = date('Y-m-d H:i:s');
        $end_date = date('Y-m-d H:i:s', strtotime("+{$duration_days} days"));
        
        // Check existing subscription
        $existing = $this->get_admin_subscription($admin_id);
        
        if ($existing && strtotime($existing['end_date']) > time()) {
            // Extend existing subscription
            $new_end_date = date('Y-m-d H:i:s', strtotime($existing['end_date'] . " +{$duration_days} days"));
            $data = [
                'end_date' => $new_end_date,
                'status' => 'active'
            ];
            return $this->update_subscription($existing['id_subscription'], $data);
        } else {
            // Create new subscription
            $data = [
                'id_admin' => $admin_id,
                'status' => 'active',
                'start_date' => $current_date,
                'end_date' => $end_date,
                'payment_method' => 'midtrans',
                'payment_ref' => 'CHAT-' . time()
            ];
            return $this->create_subscription($data);
        }
    }
    
    /**
     * Check if chat feature is active
     */
    public function is_chat_active($admin_id) {
        return $this->check_admin_subscription($admin_id);
    }
    
    /**
     * Count total payments amount
     */
    public function get_total_payments_amount($admin_id) {
        $this->db->select_sum('gross_amount');
        $this->db->where('id_admin', $admin_id);
        $this->db->where('transaction_status', 'settlement');
        $result = $this->db->get($this->table_payment)->row_array();
        return $result['gross_amount'] ?: 0;
    }
}