<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Subscription_model extends CI_Model
{
    protected $table_subscription = 'admin_subscription';
    
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }
    
    /**
     * Get admin subscription
     */
    public function get_subscription($admin_id)
    {
        $this->db->where('id_admin', $admin_id);
        $this->db->order_by('end_date', 'DESC');
        $this->db->limit(1);
        return $this->db->get($this->table_subscription)->row();
    }
    
    /**
     * Check if subscription is active
     */
    public function is_subscription_active($admin_id)
    {
        $subscription = $this->get_subscription($admin_id);
        
        if (!$subscription) {
            return false;
        }
        
        $now = date('Y-m-d H:i:s');
        $end_date = $subscription->end_date;
        
        return $subscription->status === 'active' && $end_date > $now;
    }
    
    /**
     * Activate subscription after successful payment
     */
    public function activate_subscription($admin_id, $payment_data)
    {
        // Calculate subscription period (30 days from now)
        $start_date = date('Y-m-d H:i:s');
        $end_date = date('Y-m-d H:i:s', strtotime('+30 days'));
        
        $subscription_data = [
            'id_admin' => $admin_id,
            'status' => 'active',
            'start_date' => $start_date,
            'end_date' => $end_date,
            'payment_method' => $payment_data['payment_type'] ?? 'midtrans',
            'payment_ref' => $payment_data['order_id'] ?? 'PAY-' . time(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Check if subscription exists
        $existing = $this->get_subscription($admin_id);
        
        if ($existing) {
            // Update existing subscription
            $this->db->where('id_admin', $admin_id);
            $this->db->update($this->table_subscription, $subscription_data);
        } else {
            // Create new subscription
            $this->db->insert($this->table_subscription, $subscription_data);
        }
        
        return $this->db->affected_rows() > 0;
    }
    
    /**
     * Extend subscription
     */
    public function extend_subscription($admin_id, $days = 30)
    {
        $subscription = $this->get_subscription($admin_id);
        
        if (!$subscription) {
            return false;
        }
        
        $current_end = $subscription->end_date;
        $new_end = date('Y-m-d H:i:s', strtotime($current_end . " +{$days} days"));
        
        $this->db->where('id_admin', $admin_id);
        $this->db->update($this->table_subscription, [
            'end_date' => $new_end,
            'status' => 'active'
        ]);
        
        return $this->db->affected_rows() > 0;
    }
    
    /**
     * Cancel subscription
     */
    public function cancel_subscription($admin_id)
    {
        $this->db->where('id_admin', $admin_id);
        $this->db->update($this->table_subscription, [
            'status' => 'expired',
            'end_date' => date('Y-m-d H:i:s')
        ]);
        
        return $this->db->affected_rows() > 0;
    }
    
    /**
     * Get subscription days remaining
     */
    public function get_days_remaining($admin_id)
    {
        $subscription = $this->get_subscription($admin_id);
        
        if (!$subscription || $subscription->status !== 'active') {
            return 0;
        }
        
        $now = new DateTime();
        $end = new DateTime($subscription->end_date);
        
        if ($end < $now) {
            return 0;
        }
        
        $interval = $now->diff($end);
        return $interval->days;
    }
}